<?php
class Database {
    private $pdo;

    public function __construct() {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ];
        try {
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            die("Connection failed: " . $e->getMessage());
        }
    }

    public function query($sql, $params = []) {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    }
    
    // --- User Methods ---
    public function getUser($chat_id) {
        $stmt = $this->query("SELECT * FROM users WHERE chat_id = ?", [$chat_id]);
        return $stmt->fetch();
    }
    
    public function getAllUsers() {
        $stmt = $this->query("SELECT chat_id FROM users");
        return $stmt->fetchAll();
    }

    public function createUser($chat_id, $first_name, $referrer_id = null) {
        if (!$this->getUser($chat_id)) {
            $this->query("INSERT INTO users (chat_id, first_name, referrer_id, balance) VALUES (?, ?, ?, 0)", [$chat_id, $first_name, $referrer_id]);
            return true;
        }
        return false;
    }

    public function setStep($chat_id, $step) {
        $this->query("UPDATE users SET step = ? WHERE chat_id = ?", [$step, $chat_id]);
    }

    // اضافه کردن موجودی
    public function addBalance($chat_id, $amount) {
        if ($this->getUser($chat_id)) {
            $this->query("UPDATE users SET balance = balance + ? WHERE chat_id = ?", [$amount, $chat_id]);
            return true;
        }
        return false;
    }

    // کسر موجودی (برای برداشت)
    public function deductBalance($chat_id, $amount) {
        $this->query("UPDATE users SET balance = balance - ? WHERE chat_id = ?", [$amount, $chat_id]);
    }

    // --- Transaction Methods ---
    
    // ایجاد درخواست برداشت
    public function createWithdrawRequest($chat_id, $amount, $wallet) {
        $this->query("INSERT INTO transactions (chat_id, amount, wallet, status) VALUES (?, ?, ?, 'pending')", 
                     [$chat_id, $amount, $wallet]);
        return $this->pdo->lastInsertId();
    }

    // دریافت اطلاعات تراکنش
    public function getTransaction($id) {
        $stmt = $this->query("SELECT * FROM transactions WHERE id = ?", [$id]);
        return $stmt->fetch();
    }

    // آپدیت وضعیت تراکنش
    public function updateTransactionStatus($id, $status) {
        $this->query("UPDATE transactions SET status = ? WHERE id = ?", [$status, $id]);
    }

    // --- Statistics ---
    public function getUserCount() {
        $stmt = $this->query("SELECT COUNT(*) as count FROM users");
        return $stmt->fetch()['count'];
    }

    public function getTotalBalance() {
        $stmt = $this->query("SELECT SUM(balance) as total FROM users");
        $res = $stmt->fetch()['total'];
        return $res ? $res : 0;
    }

    public function getReferralCount($chat_id) {
        $stmt = $this->query("SELECT COUNT(*) as count FROM users WHERE referrer_id = ?", [$chat_id]);
        return $stmt->fetch()['count'];
    }

    // --- Settings ---
    public function getSetting($key) {
        $stmt = $this->query("SELECT value FROM settings WHERE key_name = ?", [$key]);
        $res = $stmt->fetch();
        return $res ? $res['value'] : null;
    }

    public function updateSetting($key, $value) {
        $sql = "INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?";
        $this->query($sql, [$key, $value, $value]);
    }
    
    // این تابع را به کلاس Database اضافه کنید
    public function getTotalWithdrawals($chat_id) {
        // جمع زدن ستون amount در جدول transactions فقط برای تراکنش‌های approved
        $stmt = $this->query("SELECT SUM(amount) as total FROM transactions WHERE chat_id = ? AND status = 'approved'", [$chat_id]);
        $res = $stmt->fetch();
        // اگر نتیجه نال بود (هنوز برداشتی نداشت)، صفر برگردان
        return $res['total'] ? $res['total'] : 0;
    }
    
}